#include "StrategyManager.h"
#include "Config.h"
#include "InformationManager.h"
#include "BuildingManager.h"
#include "UnitProbeInterface.h"
#include "UnitProbeManager.h"
#include "Juno.h"
#include <assert.h>
#include "DrawManager.h"
#include "Timer.h"
#include "UnitZealotManager.h"

extern HANDLE hthread_search = nullptr;

/* try to adjust unit, building, production, etc */
void StrategyManager::onFrame()
{
	timer_display_y += 10;

	if (!(BWAPI::Broodwar->getFrameCount() == 0 ||
		BWAPI::Broodwar->getFrameCount() - last_frame > Config::StrategyManager::FrequencyFrame))
		return;

	// before map analyzed, do nothing 
	if (!InformationManager::getInstance().isMapAnalyzed())
	{
		for (auto& u : BuildingManager::getInstance().getMapUnAnalyzeBuildings())
		{
			if (u->getType().isResourceDepot() &&
				u->isIdle() &&
				BWAPI::Broodwar->self()->minerals() >= BWAPI::UnitTypes::Protoss_Probe.mineralPrice())
			{
				u->train(BWAPI::UnitTypes::Protoss_Probe);
			}
		}
		return;
	}

	if (timer.getNoStopElapsedTimeInMicroSec() + Config::Timer::StrategyManagerMaxMicroSec > Config::Timer::TotalMicroSec)
		return;
	Timer tt; 
	tt.start(); 

	last_frame = BWAPI::Broodwar->getFrameCount(); 

	// after map analyze, first recover informaion mananger, building mananger and unit manager
	recoverBeforeMapAnalyzeUnits();

	InformationManager&		IM = InformationManager::getInstance();
	BuildingManager&		BM = BuildingManager::getInstance();
	UnitProbeManager&		UPM = UnitProbeManager::getInstance();

	// 0 depot is destroy, try first to rebuild 
	rebuildDepot(); 

	// 1.1 DEFEND->MINERAL 
	swapRegionDefendMineral(IM.getSelfRegion()); 

	/* 1.2 too many DEFENDs */
	makeRegionDefendToMineral(IM.getSelfRegion()); 

	/* 2.1 MINERAL->DEFEND, enemy more than self DEFEND */
	makeRegionMineralToDefend(IM.getSelfRegion()); 

	/* 2.2 MINERAL very close to enemy */
//	makeRegionMineralFastAttack(IM.getSelfRegion()); 

	/* 3 ATTACK->SCOUT */
	for (auto& r : BWTA::getRegions())
		makeRegionAttackToMineral(r); 
	
	/* 4 MINERAL -> SCOUT */
	makeRegionMineralToScout(IM.getSelfRegion()); 
	
	/* 5 SCOUT -> ATTACK */
	makeScoutToAttack(); 

	updateZealotTargetRegion(); 

	/* replenish minimum attack */
	for (auto& r : BWTA::getRegions())
		replenishMinimumAttack(r); 
	
	// 6 MINERAL->ATTACK 
//	makeMineralToAttack(IM.getSelfRegion()); 

	// 7 supply limited 
//	replenishSupply(); 

	// 7-1 recover power
	recoverBuildPower(); 

	double t0 = tt.getNoStopElapsedTimeInMicroSec(); 

	// update build actions
	updateBuildActions(); 	
	if (is_updated_build_actions)
	{
		build_actions = update_build_actions; 
		is_updated_build_actions = false; 
		last_action_update_frame = BWAPI::Broodwar->getFrameCount(); 
		if (hthread_search)
		{
			CloseHandle(hthread_search); 
			hthread_search = nullptr; 
		}
	}
	double t1 = tt.getNoStopElapsedTimeInMicroSec(); 
	
	// 8 follow build actions
	executeBuildActions(); 

	double t2 = tt.getNoStopElapsedTimeInMicroSec(); 

	DrawManager::getInstance().drawSearchBuild(build_actions);

	tt.stop();
	DrawManager::getInstance().drawTime(10, timer_display_y, Config::StrategyManager::FrequencyFrame,
		"StrategyManager", tt.getElapsedTimeInMicroSec());

}


/* sort compare function
sort unit from hp low to hp high */
bool StrategyManager::compUnitHpShield(const UnitProbe& u1, const UnitProbe& u2)
{
	/* hp+shield compare: less health */
	if ((u1->getUnit()->getHitPoints() + u1->getUnit()->getShields())
		< (u2->getUnit()->getHitPoints() + u2->getUnit()->getShields()))
		return true;
	else
		return false;
}


/* replace low hp defend units in self region by mineral units */
void StrategyManager::swapRegionDefendMineral(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance(); 

	// sort mineral units
	std::vector<UnitProbe> mu_low_high = IM.getRegionSelfUnits(region, Role::MINERAL);
	std::sort(mu_low_high.begin(), mu_low_high.end(), compUnitHpShield);
	std::vector<UnitProbe>::reverse_iterator mu_riter = mu_low_high.rbegin();

	// sort defend units
	std::vector<UnitProbe> du_low_high = IM.getRegionSelfUnits(IM.getSelfRegion(), Role::DEFEND);
	std::sort(du_low_high.begin(), du_low_high.end(), compUnitHpShield);
	std::vector<UnitProbe>::iterator du_iter = du_low_high.begin();

	for (; du_iter != du_low_high.end() && mu_riter != mu_low_high.rend();)
	{
		int mhp = (*mu_riter)->getUnit()->getHitPoints() + (*mu_riter)->getUnit()->getShields();
/*		int mhp_max = (*mu_riter)->getUnit()->getType().maxHitPoints() + (*mu_riter)->getUnit()->getType().maxShields();
		if (mhp < 0.7*mhp_max)
			break; */

		int dhp = (*du_iter)->getUnit()->getHitPoints() + (*du_iter)->getUnit()->getShields();
/*		int dhp_max = (*du_iter)->getUnit()->getType().maxHitPoints() + (*du_iter)->getUnit()->getType().maxShields();
		if (dhp < 0.4*dhp_max) */
		if (mhp > dhp)
		{
			UnitProbeManager::getInstance().swap(*du_iter, *mu_riter);
			du_iter++;
			mu_riter++;
		}
		else
			break;
	}
}

/* let extra defend units in a region to mineral */
void StrategyManager::makeRegionDefendToMineral(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance();
	BWAPI::WeaponType wt = BWAPI::UnitTypes::Protoss_Probe.groundWeapon(); 

	double enemy_weapon_sum = IM.getRegionEnemyWeaponSum(region);
	double self_weapon_sum = IM.getRegionSelfWeaponSum(region);
	int num = (self_weapon_sum - enemy_weapon_sum) /
		(double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown()); 

	if (num > 0)
	{
		std::vector<UnitProbe> du_low_high = IM.getRegionSelfUnits(region, Role::DEFEND);
		std::sort(du_low_high.begin(), du_low_high.end(), compUnitHpShield);
		std::vector<UnitProbe>::iterator du_iter = du_low_high.begin();
		for (int i = 0; i < num && du_iter != du_low_high.end(); i++, du_iter++)
		{
			(*du_iter)->clear();
			(*du_iter)->setRole(Role::MINERAL);
			(*du_iter)->setTargetRegion(region);
		}
	}
}

/* let mineral to defend if enemy is more than self */
void StrategyManager::makeRegionMineralToDefend(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance();
	BWAPI::WeaponType wt = BWAPI::UnitTypes::Protoss_Probe.groundWeapon();

	double enemy_weapon_sum = IM.getRegionEnemyWeaponSum(region);
	double self_weapon_sum = IM.getRegionSelfWeaponSum(region);

	int num = __max(0.0, std::ceil((enemy_weapon_sum - self_weapon_sum) /
		(double(wt.damageAmount()) * wt.damageFactor() / wt.damageCooldown()))); 

	if (num > 0)
	{
		std::vector<UnitProbe> mu_low_high = IM.getRegionSelfUnits(region, Role::MINERAL);
		num = __max(0, __min(num, int(mu_low_high.size() - 1))); 
		std::sort(mu_low_high.begin(), mu_low_high.end(), compUnitHpShield); 
		std::vector<UnitProbe>::reverse_iterator mu_riter = mu_low_high.rbegin();
		for (int i = 0; i < num && mu_riter != mu_low_high.rend(); i++, mu_riter++)
		{
			(*mu_riter)->clear();
			(*mu_riter)->setRole(Role::DEFEND);
			(*mu_riter)->setTargetRegion(region);
		}
	}
}

/* let mineral attack very close enemy */
void StrategyManager::makeRegionMineralFastAttack(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance();
	int probe_max_hp = BWAPI::UnitTypes::Protoss_Probe.maxHitPoints() + BWAPI::UnitTypes::Protoss_Probe.maxShields(); 

	std::vector<UnitProbe> munits = IM.getRegionSelfUnits(region, Role::MINERAL); 

	if (munits.size() <= Config::StrategyManager::MinMineNum)
		return; 

	int available = munits.size() - Config::StrategyManager::MinMineNum;

	for (auto& u : munits)
	{
		if (available <= 0)
			break; 

		if (u->getUnit()->getGroundWeaponCooldown()>0 ||
			(u->getUnit()->getHitPoints() + u->getUnit()->getShields()) < 0.7*probe_max_hp)
			continue;
		BWAPI::Unit eu = u->getUnit()->getClosestUnit(BWAPI::Filter::IsEnemy && !BWAPI::Filter::IsFlying, 1.5 * 32); 
		if (eu)
		{
			u->clear();
			u->setTargetRegion(region);
			u->setRole(Role::DEFEND);
			available++; 
		}
	}
}

/* let attack after combat to scout */
void StrategyManager::makeRegionAttackToMineral(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance();

	if (IM.getRegionSelfUnits(region, Role::ATTACK).empty() ||
		!IM.getRegionEnemyBuildings(region).empty() || !IM.getRegionEnemyUnits(region).empty())
		return; 

	for (auto& u : IM.getRegionSelfUnits(region, Role::ATTACK))
	{
		// remember to delete InformationManager data 
		IM.removeRegionSelfUnit(u); 
		u->clear(); 
		u->setRole(Role::MINERAL); 
		u->setTargetRegion(IM.getSelfRegion());
		u->setTargetBaseLocation(IM.getSelfBaseLocation());
		IM.addRegionSelfUnit(u); 
	}
}


/* let more mineral to scout */
void StrategyManager::makeRegionMineralToScout(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance();

	// region has no depot (maybe destroyed), wait for it
	if (IM.getRegionSelfBuildings(region, BWAPI::UnitTypes::Protoss_Nexus).empty())
		return; 

	if (!IM.isMapAnalyzed())
		return; 

	// scout enemy start location 
	if (!IM.isEnemyStartLocationFound() && IM.getStartLocationsScouts().empty())
	{
		if (BWAPI::Broodwar->getFrameCount() < 6 * 42)
			return;

		bool flag_need = true; 
		for (auto& u : IM.getRegionSelfUnits(IM.getSelfRegion(), Role::MINERAL))
		{
			if (!u->getUnit()->isCarryingMinerals())
			{
				flag_need = false; 
				IM.removeRegionSelfUnit(u); 
				u->clear(); 
				u->setRole(Role::SCOUT); 
				IM.addTileScout(u); 
				break; 
			}
		}
		if (flag_need && !IM.getRegionSelfUnits(IM.getSelfRegion(), Role::MINERAL).empty())
		{
			UnitProbe u = IM.getRegionSelfUnits(IM.getSelfRegion(), Role::MINERAL)[0]; 
			IM.removeRegionSelfUnit(u);
			u->clear();
			u->setRole(Role::SCOUT);
			IM.addTileScout(u);
		}
	}

	// mineral patches
/*	int minerals = 0; 
	for (auto& bl : region->getBaseLocations())
		minerals += bl->getMinerals().size();

	// mineral units
	std::vector<UnitProbe> mineral_units = IM.getRegionSelfUnits(region, Role::MINERAL);
	int mu_num = mineral_units.size(); 
	if (mu_num <= 2.5*minerals)
		return; 

	int num = mu_num - 2.5*minerals;
	// here I just let unit become SCOUT, 
	// their further type (STARTLOCATION/BASELOCATION) and target is determined in other function
	for (auto& u : mineral_units)
	{
		if (num-- < 0)
			break; 
		// remember to delete InformationManager data 
		IM.removeRegionSelfUnit(u);
		u->clear(); 
		u->setRole(Role::SCOUT); 
		IM.addTileScout(u); 
	} */
}


/* let scout to assist attack region */
void StrategyManager::makeScoutToAttack()
{
	InformationManager& IM = InformationManager::getInstance();
	if (!IM.isEnemyStartLocationFound())
		return; 

	// get scout units 
	std::vector<UnitProbe> scouts_start = IM.getStartLocationsScouts(); 
	std::vector<UnitProbe> scouts_base = IM.getBaseLocationsScouts(); 
	std::vector<UnitProbe> scouts_tile = IM.getTileScouts(); 
	int scout_num = scouts_start.size() + scouts_base.size() + scouts_tile.size(); 

	if (scout_num == 0)
		return; 

	std::multimap<int, BWTA::Region*> attacks_regions;
	int attack_num = 0; 
	for (auto& r : BWTA::getRegions())
	{
		// no attack in self region
		if (r == IM.getSelfRegion())
			continue; 
		// not attack region 
		if (IM.getRegionEnemyBuildings(r).empty())
			continue; 

		int num = IM.getRegionSelfUnits(r, Role::ATTACK).size(); 
		attack_num += num; 
		attacks_regions.insert(std::pair<int, BWTA::Region*>(num, r)); 
	}

	// no attack region
	if (attacks_regions.empty())
		return; 

	// get average attack num after adjust scouts_tile
	int attack_average_num = std::ceil(double(attack_num + scout_num) / attacks_regions.size()); 
	std::vector<UnitProbe>::iterator iter_start = scouts_start.begin(); 
	std::vector<UnitProbe>::iterator iter_base = scouts_base.begin(); 
	std::vector<UnitProbe>::iterator iter_tile = scouts_tile.begin(); 
	for (auto& ars : attacks_regions)
	{
		int num = __max(0, attack_average_num - ars.first); 
		while (num > 0)
		{
			UnitProbe up = nullptr; 
			if (iter_tile != scouts_tile.end())
			{
				up = *iter_tile;
				IM.removeTileScout(up); 
				iter_tile++; 
			}
			else if (iter_base != scouts_base.end())
			{
				up = *iter_base; 
				IM.removeBaseLocationScout(up); 
				iter_base++; 
			}
			else if (iter_start != scouts_start.end())
			{
				up = *iter_start; 
				IM.removeStartLocationScout(up); 
				iter_start++; 
			}

			if (up)
			{
				up->clear();
				up->setRole(Role::ATTACK);
				up->setTargetRegion(ars.second);
				IM.addRegionSelfUnit(up);
			}
			else
				break; 
			num--; 
		}
	}
}

/* let mineral to replenish attack */
void StrategyManager::makeMineralToAttack(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance();
	BuildingManager& BM = BuildingManager::getInstance(); 

	std::vector<UnitProbe> mus_low_high = IM.getRegionSelfUnits(region, Role::MINERAL);

	// special consider for zerg 
/*	if (BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Zerg)
	{
		if (mus_low_high.size() < 10)
			return; 
	}
	// other races case
	else */
	{
		// haven't reach build cannon condition
		if (BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Forge).empty())
		{
			return;
		}

		// if haven't save enough mineral for cannon, don't go
		int need_cannon = 0;
		for (auto& r : BWTA::getRegions())
		{
			need_cannon += IM.getRegionEnemyBuildings(r).size();
		}

		if (BWAPI::Broodwar->self()->minerals() < need_cannon*BWAPI::UnitTypes::Protoss_Photon_Cannon.mineralPrice())
			return;
	}
	
	// leave 3 miner at least 
	if (mus_low_high.size() <= 3)
		return; 

	std::multimap<int, BWTA::Region*> attack_regions; 
	int attack_num = 0; 
	for (auto& r : BWTA::getRegions())
	{
		// no attack in self region
		if (r == region)
			continue;
		// no attack happen
		if (IM.getRegionEnemyBuildings(r).empty())
			continue; 

		int num = IM.getRegionSelfUnits(r, Role::ATTACK).size(); 
		attack_num += num; 
		attack_regions.insert(std::pair<int, BWTA::Region*>(num, r)); 
	}

	if (attack_regions.empty())
		return; 

	// __max available or needed minerals
	int mineral_num = __max(0, int(mus_low_high.size() - 3));
	// average attackers in each region after adjusting 
	int average_num = std::ceil(double(attack_num + mineral_num) / attack_regions.size()); 

	std::sort(mus_low_high.begin(), mus_low_high.end(), compUnitHpShield);

	std::vector<UnitProbe>::reverse_iterator iter = mus_low_high.rbegin(); 

	int probe_max_hp = BWAPI::UnitTypes::Protoss_Probe.maxHitPoints() + BWAPI::UnitTypes::Protoss_Probe.maxShields();

	for (auto& ar : attack_regions)
	{
		int num = average_num - ar.first; 
		bool fail = false; 
		while (num > 0)
		{
			int hp = (*iter)->getUnit()->getHitPoints() + (*iter)->getUnit()->getShields(); 
			if (iter == mus_low_high.rend() || hp < 0.6*probe_max_hp)
			{
				fail = true; 
				break;
			}
			// remove from previous informatoinmanager
			IM.removeRegionSelfUnit(*iter); 
			// set to new role 
			(*iter)->clear(); 
			(*iter)->setRole(Role::ATTACK); 
			(*iter)->setTargetRegion(ar.second); 
			// add new informationmanager
			IM.addRegionSelfUnit(*iter); 

			iter++; 
			num--; 
		}
		if (fail)
			break;
	}
}

/* replenish supply when limited */
void StrategyManager::replenishSupply()
{
	InformationManager& IM = InformationManager::getInstance();
	BuildingManager& BM = BuildingManager::getInstance(); 

	if (BWAPI::Broodwar->self()->supplyUsed() < BWAPI::Broodwar->self()->supplyTotal() - 2)
		return;

	else if (!BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Pylon).empty())
		return; 

	int need = std::ceil(double(BWAPI::Broodwar->self()->supplyUsed() + 1 - BWAPI::Broodwar->self()->supplyTotal()) / 
		BWAPI::UnitTypes::Protoss_Pylon.supplyProvided()); 
	while (need > 0 && IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
	{
		BM.buildBasePylon(); 
		need--; 
	}
}

/* update build actions */
void StrategyManager::updateBuildActions()
{
	// still searching
	if (is_searching)
		return; 
	// still not update last searched actions
	if (is_updated_build_actions)
		return; 
/*	if (!(BWAPI::Broodwar->getFrameCount() < 5*42 ||
		BWAPI::Broodwar->getFrameCount() - last_action_update_frame > Config::StrategyManager::SearchFrequencyFrame))
		return;  */

	InformationManager& IM = InformationManager::getInstance(); 
	BuildingManager& BM = BuildingManager::getInstance(); 

	Timer timer; 
	timer.start(); 

	is_searching = false;
	is_updated_build_actions = true;

	update_build_actions.clear(); 
	// need new depot
	if (IM.getSelfBaseLocation() && 
		BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Nexus).empty() &&
		BM.getPlanningUnits(BWAPI::UnitTypes::Protoss_Nexus).empty())
	{
		int account = IM.getSelfBaseLocation()->minerals();
		if (account < 5 * BWAPI::UnitTypes::Protoss_Nexus.mineralPrice())
		{
			BWTA::BaseLocation* new_bl = getNewBaseLocation();
			if (new_bl)
			{
				update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Nexus, BWAPI::Position(new_bl->getTilePosition())));
				return;
			}
		}
	}
	// miners reach 7
	if (IM.getMineral() >= BWAPI::UnitTypes::Protoss_Probe.mineralPrice() && 
		IM.getRegionSelfUnits(IM.getSelfRegion(), Role::MINERAL).size() < Config::StrategyManager::StepOneMinerNum)
	{
		BWAPI::Unit depot = nullptr; 
		depot = (BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Nexus).empty() ? nullptr : BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Nexus)[0]); 
		if (depot)
			update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Probe, depot)); 
	}
	double t1 = timer.getNoStopElapsedTimeInMicroSec(); 
	// no forge build forge
	if (BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Forge).empty() && 
		BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Forge).empty() && 
		BM.getPlanningUnits(BWAPI::UnitTypes::Protoss_Forge).empty())
	{
		BWAPI::TilePosition location = BM.getBaseBuildLocation(BWAPI::UnitTypes::Protoss_Forge); 
		if (location.isValid() && 
			IM.getMineral() >= BWAPI::UnitTypes::Protoss_Forge.mineralPrice())
			update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Forge, BWAPI::Position(location)));
		else if (!location.isValid() && IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
		{
			location = BM.getBasePylonLocation(); 
			BM.makeLocationPower(location, BWAPI::UnitTypes::Protoss_Forge); 
		}
	}
	double t2 = timer.getNoStopElapsedTimeInMicroSec();
	// enemy region cannon
	bool need_cannon = true;
	if (IM.isEnemyStartLocationFound() && 
		(!BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Forge).empty() || 
		!BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Forge).empty() ||
		!BM.getPlanningUnits(BWAPI::UnitTypes::Protoss_Forge).empty()))
	{
		need_cannon = false; 
		for (auto& r : BWTA::getRegions())
		{
			if (r == IM.getSelfRegion() && 
				IM.getRegionEnemyUnits(r).size() >= 3 && 
				IM.getRegionSelfBuildings(r, BWAPI::UnitTypes::Protoss_Photon_Cannon).size() < 4)
			{
				BWAPI::TilePosition location = BM.getBaseBuildLocation(BWAPI::UnitTypes::Protoss_Photon_Cannon); 
				if (location.isValid() &&
					IM.getMineral() >= BWAPI::UnitTypes::Protoss_Photon_Cannon.mineralPrice())
				{
					update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Photon_Cannon, BWAPI::Position(location)));
				}
				else if (!location.isValid() &&
					IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
				{
					location = BM.getBasePylonLocation(); 
					if (location.isValid())
						update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Pylon, BWAPI::Position(location)));
				}
			}
			else if (!IM.getRegionEnemyBuildings(r).empty())
			{
				BWAPI::TilePosition location; 
				bool is_powered = BM.getCannonLocation(r, location); 
				// power pylon
				if (!location.isValid())
				{
					continue;
				}
				need_cannon = true; 
				if (!is_powered && 
					IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
				{
					BM.makeLocationPower(location, BWAPI::UnitTypes::Protoss_Photon_Cannon); 
				}
				// or cannon
				else if (is_powered && 
					IM.getMineral() >= BWAPI::UnitTypes::Protoss_Photon_Cannon.mineralPrice())
				{
					update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Photon_Cannon, BWAPI::Position(location)));
				}
			}
		}
		if (!need_cannon && 
			IM.getMineral() >= BWAPI::UnitTypes::Protoss_Probe.mineralPrice() && 
			IM.getRegionSelfUnits(IM.getSelfRegion(), Role::MINERAL).size() < 2*IM.getSelfBaseLocation()->getMinerals().size())
		{
			BWAPI::Unit depot = nullptr;
			depot = (BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Nexus).empty() ? nullptr : BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Nexus)[0]);
			if (depot)
				update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Probe, depot));
		}
	}
	double t3 = timer.getNoStopElapsedTimeInMicroSec();
	// replenish supply
	if (BWAPI::Broodwar->self()->supplyUsed() >= BWAPI::Broodwar->self()->supplyTotal() - 4 &&
		BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Pylon).empty() &&
		BM.getPlanningUnits(BWAPI::UnitTypes::Protoss_Pylon).empty() && 
		IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
	{
		BWAPI::TilePosition location = BM.getBasePylonLocation();
		if (location.isValid())
			update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Pylon, BWAPI::Position(location)));
	}
	double t4 = timer.getNoStopElapsedTimeInMicroSec();
	// produce zealot
	if (!need_cannon)
	{
		bool is_produce = false; 
		for (auto& b : BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Gateway))
		{
			if (!is_produce && b->isIdle() && IM.getMineral() >= BWAPI::UnitTypes::Protoss_Zealot)
			{
				is_produce = true; 
				b->train(BWAPI::UnitTypes::Protoss_Zealot); 
			}
		}
	}
	// build gateway 
	if (IM.isEnemyStartLocationFound() && 
		!need_cannon &&
		IM.getMineral() >= BWAPI::UnitTypes::Protoss_Gateway &&
		(BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Gateway).size() +
		BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Gateway).size() +
		BM.getPlanningUnits(BWAPI::UnitTypes::Protoss_Gateway).size()) < Config::StrategyManager::GatewayNum)
	{
		bool build_gateway = false; 
		for (auto& r : BWTA::getRegions())
		{
			if (r == IM.getSelfRegion())
				continue; 
			if (IM.getRegionEnemyBuildings(r).empty())
				continue; 
			BWAPI::TilePosition location(-1, -1); 
			bool is_powered = BM.getFrontlineLocation(r, BWAPI::UnitTypes::Protoss_Gateway, location); 
			if (!location.isValid())
				continue; 
			else if (!is_powered &&
				IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
			{
				BM.makeLocationPower(location, BWAPI::UnitTypes::Protoss_Gateway);
				build_gateway = true; 
				break; 
			}
			else if (is_powered &&
				IM.getMineral() >= BWAPI::UnitTypes::Protoss_Gateway.mineralPrice())
			{
				build_gateway = true; 
				update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Gateway, BWAPI::Position(location)));
				break; 
			}
		}
		if (!build_gateway)
		{
			BWAPI::TilePosition location = BM.getBaseBuildLocation(BWAPI::UnitTypes::Protoss_Gateway);
			if (location.isValid() &&
				IM.getMineral() >= BWAPI::UnitTypes::Protoss_Gateway.mineralPrice())
				update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Gateway, BWAPI::Position(location)));
			else if (IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
			{
				location = BM.getBasePylonLocation();
				BM.makeLocationPower(location, BWAPI::UnitTypes::Protoss_Gateway);
			}
		}
	} 
	// build gateway is delayed
	if (IM.isEnemyStartLocationFound() &&
		need_cannon &&
		BWAPI::Broodwar->getFrameCount() > Config::StrategyManager::GatewayLatestTime &&
		IM.getMineral() >= BWAPI::UnitTypes::Protoss_Gateway &&
		(BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Gateway).size() +
		BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Gateway).size() +
		BM.getPlanningUnits(BWAPI::UnitTypes::Protoss_Gateway).size()) < Config::StrategyManager::GatewayNum)
	{
		BWAPI::TilePosition location = BM.getBaseBuildLocation(BWAPI::UnitTypes::Protoss_Gateway);
		if (location.isValid() &&
			IM.getMineral() >= BWAPI::UnitTypes::Protoss_Gateway.mineralPrice())
			update_build_actions.push_back(BuildAction(BWAPI::UnitTypes::Protoss_Gateway, BWAPI::Position(location)));
		else if (IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
		{
			location = BM.getBasePylonLocation();
			BM.makeLocationPower(location, BWAPI::UnitTypes::Protoss_Gateway);
		}
	}
	double t5 = timer.getNoStopElapsedTimeInMicroSec();

}


/* follow searched build actions*/
void StrategyManager::executeBuildActions()
{
	InformationManager& IM = InformationManager::getInstance(); 
	BuildingManager& BM = BuildingManager::getInstance(); 

	for (std::vector<BuildAction>::iterator baf_iter = build_actions.begin();
		baf_iter != build_actions.end(); )
	{
		// Probe
		if (baf_iter->first == BWAPI::UnitTypes::Protoss_Probe)
		{
			if (IM.getMineral() < baf_iter->first.mineralPrice())
				baf_iter++; 
			else
			{
				if (baf_iter->second.isUnit() && baf_iter->second.getUnit()->exists() &&
					baf_iter->second.getUnit()->isIdle())
				{
					baf_iter->second.getUnit()->train(baf_iter->first);
					baf_iter = build_actions.erase(baf_iter);
				}
				else
					baf_iter++; 
			}
		}
		// pylon
		else if (baf_iter->first == BWAPI::UnitTypes::Protoss_Pylon)
		{
			if (IM.getMineral() < baf_iter->first.mineralPrice())
				baf_iter++;
			else
			{
				BWAPI::TilePosition location(baf_iter->second.getPosition());
				BM.buildAtLocation(location, baf_iter->first);
				baf_iter = build_actions.erase(baf_iter);
			}
		}
		// Forge 
		else if (baf_iter->first == BWAPI::UnitTypes::Protoss_Forge)
		{
			BWAPI::TilePosition location(baf_iter->second.getPosition()); 
			if (IM.getMineral() < baf_iter->first.mineralPrice())
				baf_iter++;
			else
			{
				BM.buildAtLocation(location, baf_iter->first);
				baf_iter = build_actions.erase(baf_iter);
			}
		}
		// Cannon 
		else if (baf_iter->first == BWAPI::UnitTypes::Protoss_Photon_Cannon)
		{
			BWAPI::TilePosition location(baf_iter->second.getPosition());
			if (IM.getMineral() < baf_iter->first.mineralPrice())
				baf_iter++;
			else
			{
				BM.buildAtLocation(location, baf_iter->first);
				baf_iter = build_actions.erase(baf_iter);
			}
		}
		// gateway 
		else if (baf_iter->first == BWAPI::UnitTypes::Protoss_Gateway)
		{
			BWAPI::TilePosition location(baf_iter->second.getPosition());
			if (IM.getMineral() < baf_iter->first.mineralPrice())
				baf_iter++;
			else
			{
				BM.buildAtLocation(location, baf_iter->first);
				baf_iter = build_actions.erase(baf_iter);
			}
		}
		// nexus 
		else if (baf_iter->first == BWAPI::UnitTypes::Protoss_Nexus)
		{
			BWAPI::TilePosition location(baf_iter->second.getPosition());
			if (IM.getMineral() < baf_iter->first.mineralPrice())
				baf_iter++;
			else
			{
				BM.buildAtLocation(location, baf_iter->first);
				baf_iter = build_actions.erase(baf_iter);
			}
		}
		else
			baf_iter++; 
	}
}


/* recover build power */
void StrategyManager::recoverBuildPower()
{
	InformationManager& IM = InformationManager::getInstance(); 
	BuildingManager& BM = BuildingManager::getInstance(); 

	for (auto& u : BM.getBuildingUnits())
	{
		if (u->getType().isPowerup() &&
			!BWAPI::Broodwar->hasPower(u->getTilePosition(), u->getType()) &&
			IM.getMineral() >= BWAPI::UnitTypes::Protoss_Pylon.mineralPrice())
		{
			BM.buildAtLocation(u->getTilePosition(), BWAPI::UnitTypes::Protoss_Pylon);
		}
	}
}

/* rebuild Depot if destroyed */
void StrategyManager::rebuildDepot()
{
	InformationManager& IM = InformationManager::getInstance();
	BuildingManager& BM = BuildingManager::getInstance();

	if (BM.getBuildingUnits(BWAPI::UnitTypes::Protoss_Nexus).empty() &&
		BM.getConstructingUnits(BWAPI::UnitTypes::Protoss_Nexus).empty() &&
		BM.getPlanningUnits(BWAPI::UnitTypes::Protoss_Nexus).empty())
	{
		BM.buildAtLocation(IM.getSelfBaseLocation()->getTilePosition(), BWAPI::UnitTypes::Protoss_Nexus);
	}
}


/* recover before map analyze units */
void StrategyManager::recoverBeforeMapAnalyzeUnits()
{
	InformationManager& IM = InformationManager::getInstance();
	UnitProbeManager& UPM = UnitProbeManager::getInstance(); 

	// unit probe manager
	std::vector<UnitProbe>& units = UnitProbeManager::getInstance().getBeforeMapAnazlyeUnits();

	for (std::vector<UnitProbe>::iterator iter = units.begin(); iter != units.end();)
	{
		if ((*iter)->getRole() == BEFORE_MAP_ANALYZED_SCOUT)
		{
			(*iter)->clear(); 
			std::pair<Role, BWAPI::Position> role_position = IM.getDesireScoutTarget((*iter)->getUnit()->getPosition()); 
			(*iter)->setRole(role_position.first);
			(*iter)->setTargetPosition(role_position.second); 
		}
		else if ((*iter)->getRole() == BEFORE_MAP_ANALYZED_MINERAL)
		{
			(*iter)->setRole(Role::MINERAL); 
			(*iter)->setTargetRegion(IM.getSelfRegion()); 
			(*iter)->setTargetBaseLocation(IM.getSelfBaseLocation()); 
			(*iter)->setTargetUnit(nullptr); 
			(*iter)->setTargetPosition(BWAPI::Position(-1, -1));
		}
		UPM.onUnitCreate(*iter); 
		IM.onUnitCreate(*iter);
		iter = units.erase(iter); 
	}

	// building manager
	std::vector<BWAPI::Unit>& buildings = BuildingManager::getInstance().getMapUnAnalyzeBuildings(); 
	for (std::vector<BWAPI::Unit>::iterator iter = buildings.begin(); iter != buildings.end();)
	{
		BuildingManager::getInstance().onUnitCreate(*iter); 
		IM.onUnitCreate(*iter); 
		iter = buildings.erase(iter); 
	}
}


/* replenish minimum attack */
void StrategyManager::replenishMinimumAttack(BWTA::Region* region)
{
	InformationManager& IM = InformationManager::getInstance(); 

	if (region == IM.getSelfRegion())
		return; 

	// haven't found enemy start, don't do
	if (!IM.isEnemyStartLocationFound())
		return; 

	// no enemy, don't do
	if (IM.getRegionEnemyBuildings(region).empty() &&
		IM.getRegionEnemyUnits(region).empty())
	{
		return; 
	}

	// have minimum attack, don't do
	bool need_attacker = true; 
	std::vector<UnitProbe> attack_units = IM.getRegionSelfUnits(region, Role::ATTACK); 
	std::vector<UnitProbe> build_units = IM.getRegionSelfUnits(region, Role::BUILD); 
	attack_units.insert(attack_units.end(), build_units.begin(), build_units.end()); 
	for (auto& u : attack_units)
	{
		if (u->getUnit()->getHitPoints() + u->getUnit()->getShields() >=
			0.5*(u->getUnit()->getType().maxHitPoints() + u->getUnit()->getType().maxShields()))
		{
			need_attacker = false; 
			break; 
		}
	}

	if (need_attacker)
	{
		int hp_best = 0; 
		UnitProbe up_best = nullptr; 
		int hp_max = BWAPI::UnitTypes::Protoss_Probe.maxHitPoints() + BWAPI::UnitTypes::Protoss_Probe.maxShields();
		for (auto& u : IM.getRegionSelfUnits(IM.getSelfRegion(), Role::MINERAL))
		{
			int hp = u->getUnit()->getHitPoints() + u->getUnit()->getShields(); 
			if (hp > hp_best)
			{
				hp_best = hp; 
				up_best = u; 
			}
			if (hp_best == hp_max)
				break; 
		}
		if (up_best)
		{
			IM.removeRegionSelfUnit(up_best);
			up_best->clear();
			up_best->setRole(Role::ATTACK);
			up_best->setTargetRegion(region);
			IM.addRegionSelfUnit(up_best);
		}
	}
}


void StrategyManager::updateZealotTargetRegion()
{
	UnitZealotManager& UZM = UnitZealotManager::getInstance(); 
	InformationManager& IM = InformationManager::getInstance(); 

	BWTA::Region* region = UZM.getTargetRegion(); 
	if (region)
	{
		if (!IM.getRegionEnemyBuildings(region).empty())
			return; 

		bool exist_enemy = false;
		for (auto& eu : IM.getRegionEnemyUnits(region))
		{
			if (!eu->isFlying())
			{
				exist_enemy = true;
				break;
			}
		}
		if (exist_enemy)
			return;
	}

	for (auto& r : BWTA::getRegions())
	{
		if (!IM.getRegionEnemyBuildings(r).empty())
		{
			UZM.setTargetRegion(r); 
			return; 
		}
	}

	BWAPI::Unit eu = BWAPI::Broodwar->getClosestUnit(IM.getSelfRegion()->getCenter(),
		BWAPI::Filter::IsEnemy && BWAPI::Filter::CanAttack);
	if (eu)
		UZM.setTargetRegion(IM.getRegion(eu->getPosition()));
	else
		UZM.setTargetRegion(nullptr); 
}


BWTA::BaseLocation* StrategyManager::getNewBaseLocation()
{
	InformationManager& IM = InformationManager::getInstance(); 

	BWTA::BaseLocation* bl_closest = nullptr; 
	double dist_closest = 1e9; 
	for (auto& bl : BWTA::getBaseLocations())
	{
		if (bl == IM.getSelfBaseLocation())
			continue; 

		BWTA::Region* r = bl->getRegion();
		if (!r->isReachable(IM.getSelfRegion()))
			continue; 
		
		if (!IM.getRegionEnemyBuildings(r).empty())
			continue; 

		int account = 0; 
		for (auto& m : bl->getStaticMinerals())
			account += m->getResources(); 
		if (account < Config::StrategyManager::BaseLocationMinMineral)
			continue; 

		int dist = IM.getSelfRegion()->getCenter().getDistance(r->getCenter()); 
		if (dist < dist_closest)
		{
			dist_closest = dist; 
			bl_closest = bl; 
		}
	}

	return bl_closest; 
}